(() => {
  let deferredPrompt;

  // Detect if the PWA can be installed
  window.addEventListener('beforeinstallprompt', (e) => {
    e.preventDefault();
    deferredPrompt = e;
    showInstallPrompt();
  });

  // Event when the PWA has been installed
  window.addEventListener('appinstalled', () => {
    hideInstallPrompt();
  });

  // Check installation status on page load
  window.addEventListener('load', async () => {
    try {
      const isInstalled = await checkInstallation();
      isInstalled ? hideInstallPrompt() : showInstallPrompt();
    } catch (error) {
      console.error("Error checking installation status:", error);
    }
  });

  // Check if the app is already installed
  async function checkInstallation() {
    try {
      if ('getInstalledRelatedApps' in navigator) {
        const relatedApps = await navigator.getInstalledRelatedApps();
        return relatedApps.some(app => app.id === 'com.example.portalremix'); // Customize this ID
      }

      return window.matchMedia('(display-mode: standalone)').matches ||
             navigator.standalone ||
             document.referrer.includes('android-app://');
    } catch (e) {
      console.warn("Error in checkInstallation:", e);
      return false;
    }
  }

  // Execute PWA installation
  async function installPWA() {
    try {
      if (deferredPrompt) {
        deferredPrompt.prompt();
        const { outcome } = await deferredPrompt.userChoice;

        if (outcome === 'accepted') {
          setTimeout(async () => {
            if (await checkInstallation()) {
              hideInstallPrompt();
            }
          }, 1000);
        }

        deferredPrompt = null;
      } else if (isIOS()) {
        // iOS: installation is manual
      } else if (isAndroid()) {
        const installed = await checkInstallation();
        if (!installed) {
          // If not installed, do nothing
        }
      } else {
        // Show popup for unsupported browsers
        const popup = document.getElementById('popupIncompatibility');
        if (popup) {
          popup.style.display = 'block';
        }
      }

      hideInstallPrompt();
    } catch (error) {
      console.error("Error during installation:", error);
    }
  }

  // Show install prompt
  function showInstallPrompt() {
    const el = document.querySelector('.install-prompt');
    if (el) {
      checkInstallation().then(installed => {
        if (!installed) el.style.display = 'flex';
      }).catch(err => {
        console.warn("Failed to show install prompt:", err);
      });
    }
  }

  // Hide install prompt
  function hideInstallPrompt() {
    const el = document.querySelector('.install-prompt');
    if (el) el.style.display = 'none';
  }

  // Detect iOS
  function isIOS() {
    const userAgent = navigator.userAgent.toLowerCase();
    return /iphone|ipad|ipod/.test(userAgent) || (userAgent.includes('mac') && 'ontouchend' in document);
  }

  // Detect Android
  function isAndroid() {
    return /android/i.test(navigator.userAgent);
  }

  // Close popup
  function closePopup() {
    const popup = document.getElementById("popupIncompatibility");
    if (popup) popup.style.display = "none";
  }

  // Show popup with custom message
  function showPopup(message) {
    const popup = document.getElementById("popupIncompatibility");
    const messageEl = document.getElementById("popupMessage");
    if (popup && messageEl) {
      messageEl.innerText = message;
      popup.style.display = "block";
    }
  }

  // Initialize install button
  document.addEventListener('DOMContentLoaded', () => {
    const installButton = document.querySelector('.install-prompt button');
    if (installButton) {
      installButton.addEventListener('click', installPWA);
    }
  });

  // Expose necessary global functions
  window.closePopup = closePopup;
  window.showPopup = showPopup;
})();
