// Configuración principal de la radio
const radioConfig = {
  streamUrl: 'https://hirschmilch.de:7000/stream',
  streamInfoUrl: 'https://app0102.sonicpanelradio.com/cp/get_info.php?p=8404',
  defaultCover: 'img/logo.png',
  stationLogo: 'img/envivo.gif',
  stationName: 'Radio Max',
  stationSlogan: 'La Radio Que Toca Tu Corazon',
  websiteUrl: 'https://radiomax.wixsite.com/radiomax',
  facebookUrl: 'https://www.facebook.com/radiomax/',
  twitterUrl: 'https://x.com/radiomax',
  whatsappNumber: '56947935574',
  instagramUrl: '',
  tiktokUrl: ''
};

let audio;
let isPlaying = false;
let currentSong = '';

document.addEventListener('DOMContentLoaded', () => {
  updateSocialLinks();
});

// Inicializa el reproductor y sus eventos
function initPlayer() {
  setInitialImages();
  updateSocialLinks();

  audio = new Audio(radioConfig.streamUrl);
  audio.crossOrigin = "anonymous";
  audio.preload = 'auto';

  const title = document.getElementById('title');
  const artist = document.getElementById('artist');
  if (title) title.textContent = radioConfig.stationName;
  if (artist) artist.textContent = 'En vivo';

  const playBtn = document.querySelector('.play-btn');
  if (playBtn) playBtn.addEventListener('click', togglePlay);

  // Configura Media Session API
  if ('mediaSession' in navigator) {
    navigator.mediaSession.metadata = new MediaMetadata({
      title: radioConfig.stationName,
      artist: 'En vivo',
      artwork: [{ src: radioConfig.defaultCover, sizes: '512x512', type: 'image/png' }]
    });

    navigator.mediaSession.setActionHandler('play', () => !isPlaying && togglePlay());
    navigator.mediaSession.setActionHandler('pause', () => isPlaying && togglePlay());
  }

  updateNowPlaying();
  setInterval(updateNowPlaying, 10000);
}

window.addEventListener('DOMContentLoaded', initPlayer);

// Controla la reproducción
function togglePlay() {
  if (!audio) return;
  const btn = document.querySelector('.play-btn');

  if (isPlaying) {
    audio.pause();
    isPlaying = false;
    if (btn) {
      btn.innerHTML = '<i class="fas fa-play"></i>';
      btn.classList.remove('playing');
    }
  } else {
    audio.play().catch(console.error);
    isPlaying = true;
    if (btn) {
      btn.innerHTML = '<i class="fas fa-pause"></i>';
      btn.classList.add('playing');
    }
  }
}

// Actualiza título y carátula usando get_info + iTunes
async function updateNowPlaying() {
  const titleEl = document.querySelector('.track-title');
  const artistEl = document.querySelector('.artist');

  try {
    const infoResponse = await fetch(radioConfig.streamInfoUrl);
    const infoData = await infoResponse.json();
    const songTitle = infoData?.title || radioConfig.stationName;
    let coverArt = infoData?.art || '';
    let foundFromItunes = false;

    // Si no hay coverArt o es genérico, busca en iTunes como respaldo
    if (!coverArt || coverArt.includes('nodj') || coverArt.endsWith('.gif')) {
      const apiUrl = `https://api.streamafrica.net/new.search.php?query=${encodeURIComponent(songTitle)}&service=itunes`;
      const itunesResponse = await fetch(apiUrl);
      const itunesData = await itunesResponse.json();

      if (itunesData?.results?.length > 0) {
        const result = itunesData.results[0];
        coverArt = result.artworkUrl100?.replace('100x100', '512x512') || radioConfig.defaultCover;
        foundFromItunes = true;
      } else {
        coverArt = radioConfig.defaultCover;
      }
    }

    if (titleEl) titleEl.textContent = songTitle;
    if (artistEl) artistEl.textContent = foundFromItunes ? 'iTunes' : radioConfig.stationSlogan;

    updateArtwork(coverArt);
    updateMediaSession(songTitle, foundFromItunes ? 'iTunes' : radioConfig.stationSlogan, coverArt);

  } catch (error) {
    console.error('Error obteniendo información de la canción:', error);
    if (titleEl) titleEl.textContent = radioConfig.stationName;
    if (artistEl) artistEl.textContent = radioConfig.stationSlogan;
    updateArtwork(radioConfig.defaultCover);
    updateMediaSession(radioConfig.stationName, radioConfig.stationSlogan, radioConfig.defaultCover);
  }
}

// Actualiza Media Session
function updateMediaSession(title, artist, artUrl) {
  if ('mediaSession' in navigator) {
    navigator.mediaSession.metadata = new MediaMetadata({
      title,
      artist,
      artwork: [{ src: artUrl, sizes: '512x512', type: 'image/png' }]
    });
  }
}

// Establece imágenes iniciales
function setInitialImages() {
  const logo = document.querySelector('.station-logo img');
  if (logo) logo.src = radioConfig.stationLogo;

  const cover = document.querySelector('.cover-art img');
  if (cover) cover.src = radioConfig.defaultCover;

  const bg = document.querySelector('.background');
  if (bg) bg.style.backgroundImage = `url(${radioConfig.defaultCover})`;
}

// Carga imagen de portada
function updateArtwork(url) {
  const img = new Image();
  img.crossOrigin = "anonymous";
  img.onload = () => {
    const cover = document.querySelector('.cover-art img');
    if (cover) cover.src = url;

    updateThemeColors(img);

    const bg = document.querySelector('.background');
    if (bg) bg.style.backgroundImage = `url(${url})`;
  };
  img.onerror = () => {
    const fallback = radioConfig.defaultCover;
    const cover = document.querySelector('.cover-art img');
    if (cover) cover.src = fallback;

    const bg = document.querySelector('.background');
    if (bg) bg.style.backgroundImage = `url(${fallback})`;
  };
  img.src = url;
}

// Extrae color dominante de la carátula
function getAverageColor(img) {
  try {
    const colorThief = new ColorThief();
    return img.complete ? colorThief.getColor(img) : [128, 0, 255];
  } catch {
    return [128, 0, 255];
  }
}

// Aplica colores al tema
function updateThemeColors(img) {
  const [r, g, b] = getAverageColor(img);
  document.documentElement.style.setProperty('--accent-color', `rgb(${r}, ${g}, ${b})`);
  document.documentElement.style.setProperty('--accent-rgb', `${r}, ${g}, ${b}`);
  document.documentElement.style.setProperty('--background-color', `rgb(${r * 0.1}, ${g * 0.1}, ${b * 0.1})`);

  const btn = document.querySelector('.play-btn');
  if (btn) btn.style.backgroundColor = `rgb(${r}, ${g}, ${b})`;
}

// Genera enlaces sociales
function updateSocialLinks() {
  const el = document.querySelector('.social-links');
  if (!el) return;

  const links = [];
  const icons = {
    facebook: 'fab fa-facebook-f',
    twitter: 'fa-brands fa-x-twitter',
    whatsapp: 'fab fa-whatsapp',
    instagram: 'fab fa-instagram',
    tiktok: 'fab fa-tiktok',
    website: 'fas fa-globe'
  };

  if (radioConfig.facebookUrl) links.push(createSocialLink('facebook', radioConfig.facebookUrl, 'Facebook'));
  if (radioConfig.twitterUrl) links.push(createSocialLink('twitter', radioConfig.twitterUrl, 'Twitter'));
  if (radioConfig.whatsappNumber) links.push(createSocialLink('whatsapp', `https://wa.me/${radioConfig.whatsappNumber}`, 'WhatsApp'));
  if (radioConfig.instagramUrl) links.push(createSocialLink('instagram', radioConfig.instagramUrl, 'Instagram'));
  if (radioConfig.tiktokUrl) links.push(createSocialLink('tiktok', radioConfig.tiktokUrl, 'TikTok'));
  if (radioConfig.websiteUrl) links.push(createSocialLink('website', radioConfig.websiteUrl, 'Sitio web'));

  el.innerHTML = links.join('');
  el.style.display = links.length ? 'flex' : 'none';

  function createSocialLink(platform, url, label) {
    const safeUrl = sanitizeUrl(url);
    return `<a href="${safeUrl}" target="_blank" rel="noopener noreferrer" class="social-btn ${platform}" aria-label="${label}">
      <i class="${icons[platform]}"></i>
    </a>`;
  }

  function sanitizeUrl(url) {
    try {
      const parser = document.createElement('a');
      parser.href = url;
      if (parser.protocol === "http:" || parser.protocol === "https:") {
        return url;
      }
    } catch (e) {}
    return "#";
  }
}

// Registro del service worker para PWA
if ('serviceWorker' in navigator) {
  window.addEventListener('load', () => {
    navigator.serviceWorker.register('sw.js')
      .then(reg => console.log('Service Worker registrado:', reg))
      .catch(err => console.error('Error en Service Worker:', err));
  });
}

// Agrega estilos programáticamente
const style = document.createElement('style');
style.textContent = `
  html, body {
    margin: 0;
    padding: 0;
    width: 100vw;
    height: 100vh;
    overflow: hidden;
  }
  .background {
    position: absolute;
    top: 0; left: 0;
    width: 100%; height: 100%;
    background-size: cover;
    background-position: center;
    filter: blur(10px) brightness(0.6);
    transition: background-image 0.5s ease;
    z-index: 0;
    animation: zoomPan 60s ease-in-out infinite;
  }
`;
document.head.appendChild(style);

